/*
 * Disk.java
 *
 * Created on May 31, 2006, 9:03 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package ags.disk;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * Disk image base class, can be subclassed to support different sector ordering variants
 * @author blurry
 */
public abstract class Disk {
    
    /**
     * Number of sectors per track (16)
     */
    static final int SECTORS_PER_TRACK = 16;
    /**
     * Image of raw disk data
     */
    Map<Integer, Map<Integer, byte[]>> diskImage;
    
    
    /**
     * Creates a new instance of Disk
     * @param file File path of disk image to load
     * @throws java.io.IOException If the disk image could not be read
     */
    public Disk(String file) throws IOException {
        System.out.println("Opening disk: "+file);
        diskImage = new HashMap<Integer, Map<Integer, byte[]>>();
        openDiskImage(file);
        System.out.println("Successful!");
    }
    
    /**
     * Open disk image and read into memory
     * @param file File path of disk image to read
     * @throws java.io.IOException If disk image could not be read
     */
    private void openDiskImage(String file) throws IOException {
        File f = new File(file);
        if (!f.exists()) throw new IOException("Disk image "+file+" does not exist");
        FileInputStream fis = new FileInputStream(f);
        int counter = 0;
        while (fis.available() > 0) {
            byte[] sector = new byte[256];
            int read = fis.read(sector);
            if (read < 256) System.out.println("Read less than 256 bytes for last sector! (got "+read+" instead)");
            int sectorNumber = translateSectorNumber(counter % SECTORS_PER_TRACK);
            int trackNumber = counter / SECTORS_PER_TRACK;
            setSector(trackNumber, sectorNumber, sector);
            counter++;
        }
    }
    
    /**
     * Convert physical sector order (in file) to logical sector order (in memory)
     * @param sectorNumber Physical sector number
     * @return Locical sector number
     */
    protected abstract int translateSectorNumber(int sectorNumber);        
    
    /**
     * Get 256-byte sector data
     * @param trackNumber Track number to read
     * @param sectorNumber Sector number to read
     * @return 256-byte array
     */
    public byte[] getSector(int trackNumber, int sectorNumber) {
        Map<Integer, byte[]> track = diskImage.get(trackNumber);
        if (track == null) {
            System.out.println("Warning: Track "+trackNumber+" not found in image!");
            return null;
        }
        byte[] sector = track.get(sectorNumber);
        if (sector == null) {
            System.out.println("Warning: Track "+trackNumber+", Sector "+sectorNumber+" not found in image!");
            return null;            
        }
        return sector;
    }
    
    /**
     * Read sector into a buffer
     * @param trackNumber Track number to read
     * @param sectorNumber Sector number to read
     * @param sector Buffer to read data into
     */
    public void setSector(int trackNumber, int sectorNumber, byte[] sector) {
        Map<Integer, byte[]> track = diskImage.get(trackNumber);
        if (track == null) {
            track = new HashMap<Integer, byte[]>();
            diskImage.put(trackNumber, track);
        }
        track.put(sectorNumber, sector);
    }
}
